/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef GUI_4CHANNEL_H
#define GUI_4CHANNEL_H

#include <QButtonGroup>
#include <QPushButton>
#include <QStackedWidget>
#include <QWidget>
#include "config_handler.h"
#include "numpad.h"
#include "rf_channel.h"
#include "rf_system.h"
#include "rf_system_4channel.h"

namespace Ui {
class GUI_4channel;
}

class GUI_4channel : public QWidget
{
	Q_OBJECT

signals:
	void signal_channelInit(int intrasys_num);
	void signal_timerStart(int intrasys_num, int ms);
	void signal_timerStop(int intrasys_num);

	void signal_execute_error_clear(int intrasys_num);
	void signal_set_error_clearing_enable(int intrasys_num, bool enable);

	void signal_RF_enable(int intrasys_num, bool enable);
	void signal_setPowerWatt(int intrasys_num, double power_watt);
	void signal_setFrequency(int intrasys_num, double frequency);
	void signal_setPhase(int intrasys_num, double phase);
	void signal_setAutoPhase(int intrasys_num, double autophase);

	void signal_RCM_Off();
	void signal_RCM_USB();
//	void signal_RCM_TCP();

	void signal_set_ethercat_enable(bool enable);


public:
	explicit GUI_4channel(QWidget *parent = 0);
	~GUI_4channel();

	//
	// TODO:
	// These 2 should probably be private but currently need to be public due to the implementation of EtherCAT
	//
	RF_System_4channel *RF_system_constructor;
	ConfigHandler *config;

public slots:
	void handler_execute_restart_program();

private slots:

	//Manual Error Clearing
	void on_pushButton_clear_errors_1_clicked();

	//Channel Working
	void channel_working_handler(int channel_number, bool enabled);
	void handler_channel_initialization_complete(int intrasys_num, bool success);

	//Parameter Buttons
	void on_pushButton_power_watt_1_clicked();
	void on_pushButton_power_watt_2_clicked();
	void on_pushButton_power_watt_3_clicked();
	void on_pushButton_power_watt_4_clicked();

	void on_pushButton_frequency_1_clicked();
	void on_pushButton_frequency_2_clicked();
	void on_pushButton_frequency_3_clicked();
	void on_pushButton_frequency_4_clicked();

	void on_pushButton_phase_1_clicked();
	void on_pushButton_phase_2_clicked();
	void on_pushButton_phase_3_clicked();
	void on_pushButton_phase_4_clicked();

	void on_pushButton_autophase_1_clicked();
	void on_pushButton_autophase_2_clicked();
	void on_pushButton_autophase_3_clicked();
	void on_pushButton_autophase_4_clicked();

	void on_pushButton_RF_enable_1_clicked();
	void on_pushButton_RF_enable_2_clicked();
	void on_pushButton_RF_enable_3_clicked();
	void on_pushButton_RF_enable_4_clicked();

	//Numpad
	void on_pushButton_numpad_0_clicked();
	void on_pushButton_numpad_1_clicked();
	void on_pushButton_numpad_2_clicked();
	void on_pushButton_numpad_3_clicked();
	void on_pushButton_numpad_4_clicked();
	void on_pushButton_numpad_5_clicked();
	void on_pushButton_numpad_6_clicked();
	void on_pushButton_numpad_7_clicked();
	void on_pushButton_numpad_8_clicked();
	void on_pushButton_numpad_9_clicked();
	void on_pushButton_numpad_period_clicked();
	void on_pushButton_numpad_arrow_left_clicked();
	void on_pushButton_numpad_arrow_right_clicked();
	void on_pushButton_numpad_plus_clicked();
	void on_pushButton_numpad_minus_clicked();
	void on_pushButton_numpad_backspace_clicked();
	void on_pushButton_numpad_clear_all_clicked();
	void on_pushButton_numpad_ok_clicked();

	//Numpad Functions
	void numpad_value_confirmed_handler(QPushButton *button, double *num);
	void numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision = 2, double scale = 1, bool period_enable = true, bool sign_enable = false);

	//RCM
	void on_pushButton_RCM_1_clicked();
	void on_pushButton_RCM_2_clicked();

	//EtherCAT
	void on_pushButton_etherCAT_OFF_1_clicked();				//EtherCAT OFF
	void on_pushButton_etherCAT_ON_1_clicked();					//EtherCAT ON

	//Readings Handler
	void handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11_loss, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_ratio);
	void handler_RF_enable_get(int intrasys_num, bool RF_enabled);
	void handler_power_get(int intrasys_num, double power_dbm, double power_watt);
	void handler_frequency_get(int intrasys_num, double frequency_mhz);
	void handler_phase_get(int intrasys_num, double phase_degrees);
	void handler_error_get(int intrasys_num, quint64 error, QStringList error_messages);
	void handler_reset_detected(int intrasys_num, int channel_number);
	void handler_temperature_PA_get(int intrasys_num, double temperature);
	void handler_temperature_SG_get(int intrasys_num, double temperature);
	void handler_firmware_version_get(int intrasys_num, int channel_num, QString version);
	void handler_autophase_enable_get(int intrasys_num, bool enable);
	void handler_autophase_get(int intrasys_num, double phase_degrees);
	void handler_demodulator_phase_get(int intrasys_num, double phase_degrees);
	void handler_demodulator_enable_get(int intrasys_num, bool enable);
	void handler_clock_source_get(int intrasys_num, int source);
	void handler_channel_number_get(int intrasys_num, int chan_num);

private:

	Ui::GUI_4channel *ui;

	QTimer *shutdown_timer;

	QButtonGroup *buttongroup_RCM;
	QButtonGroup ethercat_buttonGroup;

	/* Remote Command Mode */
	void handler_RCM_mode();
	bool RCM_supported = false;

	/* EtherCAT Mode */
	void handler_ethercat_mode();

	QFrame *frame_array[CHANNELCOUNT];
	QStackedWidget *stackedWidget_phase_array[CHANNELCOUNT];
	QLabel *readings_array[CHANNELCOUNT][3];
	QLabel *demodulator_phase_array[CHANNELCOUNT];
	QPushButton *RF_enable_array[CHANNELCOUNT];
//	QPushButton *power_dbm_array[CHANNELCOUNT];
	QPushButton *power_watt_array[CHANNELCOUNT];
	QPushButton *phase_array[CHANNELCOUNT];
	QPushButton *autophase_array[CHANNELCOUNT];
	QPushButton *frequency_array[CHANNELCOUNT];
	QLabel *error_array[CHANNELCOUNT];
	QLabel *temperature_PA_array[CHANNELCOUNT];
	QLabel *temperature_SG_array[CHANNELCOUNT];
	QLabel *clock_source_array[CHANNELCOUNT];
	QLabel *channel_label_array[2*CHANNELCOUNT];

	Numpad *numpad;
	double numpad_value = 0;
	int scale_multiplier;
	int precision;

	int shutdown_counter;

	void configure_auto_error_clearing();

    QList<int> channel_numbers;
    QList<QString> firmware_versions;
};

#endif // GUI_4CHANNEL_H


